"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, /**
 * User log event listener manager
 */ "default", {
    enumerable: true,
    get: function() {
        return UserLogListenerManager;
    }
});
const _metaapiclient = /*#__PURE__*/ _interop_require_default(require("../../metaapi.client"));
const _randomstring = /*#__PURE__*/ _interop_require_default(require("randomstring"));
const _logger = /*#__PURE__*/ _interop_require_default(require("../../../logger"));
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let UserLogListenerManager = class UserLogListenerManager extends _metaapiclient.default {
    /**
   * Constructs user log event listener manager instance
   * @param {DomainClient} domainClient domain client
   */ constructor(domainClient){
        super(domainClient);
        this._domainClient = domainClient;
        this._strategyLogListeners = {};
        this._subscriberLogListeners = {};
        this._errorThrottleTime = 1000;
        this._logger = _logger.default.getLogger("UserLogListenerManager");
    }
    /**
   * Returns the dictionary of strategy log listeners
   * @returns {Object} dictionary of strategy log listeners
   */ get strategyLogListeners() {
        return this._strategyLogListeners;
    }
    /**
   * Returns the dictionary of subscriber log listeners
   * @returns {Object} dictionary of subscriber log listeners
   */ get subscriberLogListeners() {
        return this._subscriberLogListeners;
    }
    /**
   * Adds a strategy log listener
   * @param {UserLogListener} listener user log listener
   * @param {String} strategyId strategy id
   * @param {Date} [startTime] log search start time
   * @param {String} [positionId] position id filter
   * @param {'DEBUG'|'INFO'|'WARN'|'ERROR'} [level] minimum severity level
   * @param {Number} [limit] log pagination limit
   * @returns {String} strategy log listener id
   */ addStrategyLogListener(listener, strategyId, startTime, positionId, level, limit) {
        const listenerId = _randomstring.default.generate(10);
        this._strategyLogListeners[listenerId] = listener;
        this._startStrategyLogStreamJob(listenerId, listener, strategyId, startTime, positionId, level, limit);
        return listenerId;
    }
    /**
   * Adds a subscriber log listener
   * @param {UserLogListener} listener user log listener
   * @param {String} subscriberId subscriber id
   * @param {Date} [startTime] log search start time
   * @param {string} [strategyId] strategy id filter
   * @param {string} [positionId] position id filter
   * @param {'DEBUG'|'INFO'|'WARN'|'ERROR'} [level] minimum severity level
   * @param {Number} [limit] log pagination limit
   * @returns {String} subscriber log listener id
   */ addSubscriberLogListener(listener, subscriberId, startTime, strategyId, positionId, level, limit) {
        const listenerId = _randomstring.default.generate(10);
        this._subscriberLogListeners[listenerId] = listener;
        this._startSubscriberLogStreamJob(listenerId, listener, subscriberId, startTime, strategyId, positionId, level, limit);
        return listenerId;
    }
    /**
   * Removes strategy log listener by id
   * @param {String} listenerId listener id 
   */ removeStrategyLogListener(listenerId) {
        delete this._strategyLogListeners[listenerId];
    }
    /**
   * Removes subscriber log listener by id
   * @param {String} listenerId listener id 
   */ removeSubscriberLogListener(listenerId) {
        delete this._subscriberLogListeners[listenerId];
    }
    async _startStrategyLogStreamJob(listenerId, listener, strategyId, startTime, positionId, level, limit) {
        let throttleTime = this._errorThrottleTime;
        while(this._strategyLogListeners[listenerId]){
            const opts = {
                url: `/users/current/strategies/${strategyId}/user-log/stream`,
                method: "GET",
                params: {
                    startTime,
                    positionId,
                    level,
                    limit
                },
                headers: {
                    "auth-token": this._token
                },
                json: true
            };
            try {
                const packets = await this._domainClient.requestCopyFactory(opts, true);
                // stop job if user has unsubscribed in time of new packets has been received
                if (!this._strategyLogListeners[listenerId]) {
                    return;
                }
                await listener.onUserLog(packets);
                throttleTime = this._errorThrottleTime;
                if (this._strategyLogListeners[listenerId] && packets.length) {
                    startTime = new Date(new Date(packets[0].time).getTime() + 1);
                }
            } catch (err) {
                await listener.onError(err);
                if (err.name === "NotFoundError") {
                    this._logger.error(`Strategy ${strategyId} not found, removing listener ${listenerId}`);
                    delete this._strategyLogListeners[listenerId];
                } else {
                    this._logger.error(`Failed to retrieve user log stream for strategy ${strategyId}, ` + `listener ${listenerId}, retrying in ${Math.floor(throttleTime / 1000)} seconds`, err);
                    await new Promise((res)=>setTimeout(res, throttleTime));
                    throttleTime = Math.min(throttleTime * 2, 30000);
                }
            }
        }
    }
    async _startSubscriberLogStreamJob(listenerId, listener, subscriberId, startTime, strategyId, positionId, level, limit) {
        let throttleTime = this._errorThrottleTime;
        while(this._subscriberLogListeners[listenerId]){
            const opts = {
                url: `/users/current/subscribers/${subscriberId}/user-log/stream`,
                method: "GET",
                params: {
                    startTime,
                    strategyId,
                    positionId,
                    level,
                    limit
                },
                headers: {
                    "auth-token": this._token
                },
                json: true
            };
            try {
                const packets = await this._domainClient.requestCopyFactory(opts, true);
                // stop job if user has unsubscribed in time of new packets has been received
                if (!this._subscriberLogListeners[listenerId]) {
                    return;
                }
                await listener.onUserLog(packets);
                throttleTime = this._errorThrottleTime;
                if (this._subscriberLogListeners[listenerId] && packets.length) {
                    startTime = new Date(new Date(packets[0].time).getTime() + 1);
                }
            } catch (err) {
                await listener.onError(err);
                if (err.name === "NotFoundError") {
                    this._logger.error(`Subscriber ${subscriberId} not found, removing listener ${listenerId}`);
                    delete this._subscriberLogListeners[listenerId];
                } else {
                    this._logger.error(`Failed to retrieve user log stream for subscriber ${subscriberId}, ` + `listener ${listenerId}, retrying in ${Math.floor(throttleTime / 1000)} seconds`, err);
                    await new Promise((res)=>setTimeout(res, throttleTime));
                    throttleTime = Math.min(throttleTime * 2, 30000);
                }
            }
        }
    }
};

//# sourceMappingURL=data:application/json;base64,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