"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "default", {
    enumerable: true,
    get: function() {
        return MetaApiConnection;
    }
});
const _synchronizationListener = /*#__PURE__*/ _interop_require_default(require("../clients/metaApi/synchronizationListener"));
const _logger = /*#__PURE__*/ _interop_require_default(require("../logger"));
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let MetaApiConnection = class MetaApiConnection extends _synchronizationListener.default {
    /**
   * Opens the connection. Can only be called the first time, next calls will be ignored.
   * @param {string} instanceId connection instance id
   * @return {Promise} promise resolving when the connection is opened
   */ async connect(instanceId) {}
    /**
   * Closes the connection. The instance of the class should no longer be used after this method is invoked.
   * @param {string} instanceId connection instance id
   */ async close(instanceId) {}
    /**
   * Returns MetaApi account
   * @return {MetatraderAccount} MetaApi account
   */ get account() {
        return this._account;
    }
    /**
   * Returns connection application
   * @return {String} connection application
   */ get application() {
        return this._application;
    }
    /**
   * Schedules the refresh task
   * @param {string} region replica region
   */ scheduleRefresh(region) {
        if (!this._refreshTasks[region]) {
            var _this__options_connections;
            var _this__options_connections_refreshReplicasMaxDelayInMs;
            const delay = Math.random() * ((_this__options_connections_refreshReplicasMaxDelayInMs = (_this__options_connections = this._options.connections) === null || _this__options_connections === void 0 ? void 0 : _this__options_connections.refreshReplicasMaxDelayInMs) !== null && _this__options_connections_refreshReplicasMaxDelayInMs !== void 0 ? _this__options_connections_refreshReplicasMaxDelayInMs : 6 * 60 * 60 * 1000);
            this._refreshTasks[region] = setTimeout(this._refreshReplicas, delay);
        }
    }
    /**
   * Cancels the scheduled refresh task
   * @param {string} region replica region
   */ cancelRefresh(region) {
        clearTimeout(this._refreshTasks[region]);
        delete this._refreshTasks[region];
    }
    /**
   * Refreshes account replicas
   */ async _refreshReplicas() {
        Object.values(this._refreshTasks).forEach((task)=>clearTimeout(task));
        this._refreshTasks = {};
        const oldReplicas = {};
        this._account.replicas.forEach((replica)=>oldReplicas[replica.region] = replica.id);
        const newReplicas = {};
        let isAccountUpdated = false;
        try {
            await this._account.reload();
            isAccountUpdated = true;
            this._account.replicas.forEach((replica)=>newReplicas[replica.region] = replica.id);
        } catch (error) {
            if (error.name === "NotFoundError") {
                if (this._connectionRegistry) {
                    this._connectionRegistry.closeAllInstances(this._account.id);
                }
            }
        }
        if (isAccountUpdated) {
            const deletedReplicas = {};
            const addedReplicas = {};
            Object.keys(oldReplicas).forEach((key)=>{
                if (newReplicas[key] !== oldReplicas[key]) {
                    deletedReplicas[key] = oldReplicas[key];
                }
            });
            Object.keys(newReplicas).forEach((key)=>{
                if (newReplicas[key] !== oldReplicas[key]) {
                    addedReplicas[key] = newReplicas[key];
                }
            });
            if (Object.keys(deletedReplicas).length) {
                Object.values(deletedReplicas).forEach((replicaId)=>this._websocketClient.onAccountDeleted(replicaId));
            }
            if (Object.keys(deletedReplicas).length || Object.keys(addedReplicas).length) {
                newReplicas[this._account.region] = this._account.id;
                this._websocketClient.updateAccountCache(this._account.id, newReplicas);
                Object.entries(this._account.accountRegions).forEach(([region, instance])=>{
                    if (!this._options.region || this._options.region === region) {
                        this._websocketClient.ensureSubscribe(instance, 0);
                        this._websocketClient.ensureSubscribe(instance, 1);
                    }
                });
            }
        }
    }
    async _ensureSynchronized(instanceIndex, key) {
        let state = this._getState(instanceIndex);
        if (state && !this._closed) {
            try {
                const synchronizationResult = await this.synchronize(instanceIndex);
                if (synchronizationResult) {
                    state.synchronized = true;
                    state.synchronizationRetryIntervalInSeconds = 1;
                }
            } catch (err) {
                const level = this._latencyService.getSynchronizedAccountInstances(this._account.id).length ? "debug" : "error";
                this._logger[level]("MetaApi websocket client for account " + this._account.id + ":" + instanceIndex + " failed to synchronize", err);
                if (state.shouldSynchronize === key) {
                    setTimeout(this._ensureSynchronized.bind(this, instanceIndex, key), state.synchronizationRetryIntervalInSeconds * 1000);
                    state.synchronizationRetryIntervalInSeconds = Math.min(state.synchronizationRetryIntervalInSeconds * 2, 300);
                }
            }
        }
    }
    _getState(instanceIndex) {
        if (!this._stateByInstanceIndex["" + instanceIndex]) {
            this._stateByInstanceIndex["" + instanceIndex] = {
                instanceIndex,
                ordersSynchronized: {},
                dealsSynchronized: {},
                shouldSynchronize: undefined,
                synchronizationRetryIntervalInSeconds: 1,
                synchronized: false,
                lastDisconnectedSynchronizationId: undefined,
                lastSynchronizationId: undefined,
                disconnected: false
            };
        }
        return this._stateByInstanceIndex["" + instanceIndex];
    }
    _checkIsConnectionActive() {
        if (!this._opened) {
            throw new Error("This connection has not been initialized yet, please invoke await connection.connect()");
        }
        if (this._closed) {
            throw new Error("This connection has been closed, please create a new connection");
        }
    }
    /**
   * @typedef Config MetaApi options for connections
   * @property {Options} [connections] MetaApi connections options. Only for tests. Will be ignored when set in SDK
   */ /**
   * @typedef Options MetaApiConnection options
   * @property {number} [refreshReplicasMaxDelayInMs = 6 * 60 * 60 * 1000] max delay before refreshing replicas delay
   */ /**
   * Constructs MetaApi MetaTrader Api connection
   * @param {MetaApiOpts & Config} options MetaApi options
   * @param {MetaApiWebsocketClient} websocketClient MetaApi websocket client
   * @param {MetatraderAccount} account MetaTrader account id to connect to
   * @param {String} [application] application to use
   */ constructor(options, websocketClient, account, application){
        super();
        this._options = options;
        this._websocketClient = websocketClient;
        this._latencyService = websocketClient.latencyService;
        this._account = account;
        this._logger = _logger.default.getLogger("MetaApiConnection");
        this._application = application;
        this._refreshReplicas = this._refreshReplicas.bind(this);
        this._refreshTasks = {};
    }
};

//# sourceMappingURL=data:application/json;base64,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