'use strict';
/**
 * Base class for API errors. Contains indication of HTTP status.
 */ const isNode = typeof process !== 'undefined' && process.release && process.release.name === 'node';
export class ApiError extends Error {
    /**
   * Sets error code, used for i18n
   * @param {string} code error code for i18n
   */ set code(code) {
        this._code = code;
    }
    /**
   * Returns error code used for i18n
   * @return {string} error code
   */ get code() {
        return this._code;
    }
    /**
   * Set message arguments for i18n
   * @param {Array<Object>} args arguments for i18n
   */ set arguments(args) {
        this._args = args;
    }
    /**
   * Returns message arguments for i18n
   * @return {Array<Object>} message arguments for i18n
   */ get arguments() {
        return this._args;
    }
    /**
   * ApiError constructor
   * @param {Function} clazz error name
   * @param {Object} message error message
   * @param {number} status HTTP status
   * @param {string} url API request URL
   */ constructor(clazz, message, status, url){
        super(url ? message + '. Request URL: ' + url : message);
        /**
     * Error name
     * @type {string}
     */ this.name = clazz.name;
        /**
     * HTTP status code
     * @type {number}
     */ this.status = status;
        /**
     * API request URL
     */ this.url = url;
        if (isNode && Error.captureStackTrace) {
            Error.captureStackTrace(this, clazz);
        }
    }
}
/**
 * Throwing this error results in 404 (Not Found) HTTP response code.
 */ export class NotFoundError extends ApiError {
    /**
   * Represents NotFoundError.
   * @param {string} message error message
   * @param {string} url API request URL
   */ constructor(message, url){
        super(NotFoundError, message, 404, url);
    }
}
/**
 * Throwing this error results in 403 (Forbidden) HTTP response code.
 */ export class ForbiddenError extends ApiError {
    /**
   * Constructs forbidden error.
   * @param {string} message error message
   * @param {string} url API request URL
   */ constructor(message, url){
        super(ForbiddenError, message, 403, url);
    }
}
/**
 * Throwing this error results in 401 (Unauthorized) HTTP response code.
 */ export class UnauthorizedError extends ApiError {
    /**
   * Constructs unauthorized error.
   * @param {string} message error message
   * @param {string} url API request URL
   */ constructor(message, url){
        super(UnauthorizedError, message, 401, url);
    }
}
/**
 * Represents validation error. Throwing this error results in 400 (Bad Request) HTTP response code.
 */ export class ValidationError extends ApiError {
    /**
   * Constructs validation error.
   * @param {string} message error message
   * @param {Object} details error data
   * @param {string} url API request URL
   */ constructor(message, details, url){
        super(ValidationError, message, 400, url);
        /**
     * Validation error details
     * @type {Object}
     */ this.details = details;
    }
}
/**
 * Represents unexpected error. Throwing this error results in 500 (Internal Error) HTTP response code.
 */ export class InternalError extends ApiError {
    /**
   * Constructs unexpected error.
   * @param {string} message error message
   * @param {string} url API request URL
   */ constructor(message, url){
        super(InternalError, message, 500, url);
    }
}
/**
 * Represents conflict error. Throwing this error results in 409 (Conflict) HTTP response code.
 */ export class ConflictError extends ApiError {
    /**
   * Constructs conflict error.
   * @param {string} message error message
   * @param {string} url API request URL
   */ constructor(message, url){
        super(ConflictError, message, 409, url);
    }
}
/**
 * @typedef {Object} TooManyRequestsErrorMetadata
 * @property {Number} periodInMinutes throttling period in minutes
 * @property {Number} requestsPerPeriodAllowed available requests for periodInMinutes
 * @property {Date} recommendedRetryTime recommended date to retry request
 * @property {String} type error type
 */ /**
 * Represents too many requests error. Throwing this error results in 429 (Too Many Requests) HTTP response code.
 */ export class TooManyRequestsError extends ApiError {
    /**
   * Constructs too many requests error.
   * @param {string} message error message
   * @param {TooManyRequestsErrorMetadata} metadata error metadata
   */ constructor(message, metadata, url){
        super(TooManyRequestsError, message, 429, url);
        this.metadata = metadata;
    }
}

//# sourceMappingURL=data:application/json;base64,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