'use strict';
function asyncGeneratorStep(gen, resolve, reject, _next, _throw, key, arg) {
    try {
        var info = gen[key](arg);
        var value = info.value;
    } catch (error) {
        reject(error);
        return;
    }
    if (info.done) {
        resolve(value);
    } else {
        Promise.resolve(value).then(_next, _throw);
    }
}
function _async_to_generator(fn) {
    return function() {
        var self = this, args = arguments;
        return new Promise(function(resolve, reject) {
            var gen = fn.apply(self, args);
            function _next(value) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "next", value);
            }
            function _throw(err) {
                asyncGeneratorStep(gen, resolve, reject, _next, _throw, "throw", err);
            }
            _next(undefined);
        });
    };
}
import LoggerManager from '../../logger';
let SubscriptionManager = class SubscriptionManager {
    /**
   * Returns whether an account is currently subscribing
   * @param {String} accountId account id
   * @param {Number} instanceNumber instance index number
   * @returns {Boolean} whether an account is currently subscribing
   */ isAccountSubscribing(accountId, instanceNumber) {
        if (instanceNumber !== undefined) {
            return Object.keys(this._subscriptions).includes(accountId + ':' + instanceNumber);
        } else {
            for (let key of Object.keys(this._subscriptions)){
                if (key.startsWith(accountId)) {
                    return true;
                }
            }
            return false;
        }
    }
    /**
   * Returns whether an instance is in disconnected retry mode
   * @param {String} accountId account id
   * @param {Number} instanceNumber instance index number
   * @returns {Boolean} whether an account is currently subscribing
   */ isDisconnectedRetryMode(accountId, instanceNumber) {
        let instanceId = accountId + ':' + (instanceNumber || 0);
        return this._subscriptions[instanceId] ? this._subscriptions[instanceId].isDisconnectedRetryMode : false;
    }
    /**
   * Returns whether an account subscription is active
   * @param {String} accountId account id
   * @returns {Boolean} instance actual subscribe state
   */ isSubscriptionActive(accountId) {
        return !!this._subscriptionState[accountId];
    }
    /**
   * Subscribes to the Metatrader terminal events
   * @param {String} accountId id of the MetaTrader account to subscribe to
   * @param {Number} instanceNumber instance index number
   * @returns {Promise} promise which resolves when subscription started
   */ subscribe(accountId, instanceNumber) {
        this._subscriptionState[accountId] = true;
        return this._websocketClient.rpcRequest(accountId, {
            type: 'subscribe',
            instanceIndex: instanceNumber
        });
    }
    /**
   * Schedules to send subscribe requests to an account until cancelled
   * @param {String} accountId id of the MetaTrader account
   * @param {Number} instanceNumber instance index number
   * @param {Boolean} isDisconnectedRetryMode whether to start subscription in disconnected retry
   * mode. Subscription task in disconnected mode will be immediately replaced when the status packet is received
   */ scheduleSubscribe(accountId, instanceNumber, isDisconnectedRetryMode = false) {
        var _this = this;
        return _async_to_generator(function*() {
            const client = _this._websocketClient;
            let instanceId = accountId + ':' + (instanceNumber || 0);
            if (!_this._subscriptions[instanceId]) {
                _this._subscriptions[instanceId] = {
                    shouldRetry: true,
                    task: null,
                    waitTask: null,
                    future: null,
                    isDisconnectedRetryMode
                };
                let subscribeRetryIntervalInSeconds = 3;
                while(_this._subscriptions[instanceId].shouldRetry){
                    let resolveSubscribe;
                    _this._subscriptions[instanceId].task = {
                        promise: new Promise((res)=>{
                            resolveSubscribe = res;
                        })
                    };
                    _this._subscriptions[instanceId].task.resolve = resolveSubscribe;
                    // eslint-disable-next-line no-inner-declarations, complexity
                    let subscribeTask = function() {
                        var _ref = _async_to_generator(function*() {
                            try {
                                _this._logger.debug(`${accountId}:${instanceNumber}: running subscribe task`);
                                yield _this.subscribe(accountId, instanceNumber);
                            } catch (err) {
                                if (err.name === 'TooManyRequestsError') {
                                    const socketInstanceIndex = client.socketInstancesByAccounts[instanceNumber][accountId];
                                    if (err.metadata.type === 'LIMIT_ACCOUNT_SUBSCRIPTIONS_PER_USER') {
                                        _this._logSubscriptionError(accountId, `${instanceId}: Failed to subscribe`, err);
                                    }
                                    if ([
                                        'LIMIT_ACCOUNT_SUBSCRIPTIONS_PER_USER',
                                        'LIMIT_ACCOUNT_SUBSCRIPTIONS_PER_SERVER',
                                        'LIMIT_ACCOUNT_SUBSCRIPTIONS_PER_USER_PER_SERVER'
                                    ].includes(err.metadata.type)) {
                                        delete client.socketInstancesByAccounts[instanceNumber][accountId];
                                        client.lockSocketInstance(instanceNumber, socketInstanceIndex, _this._websocketClient.getAccountRegion(accountId), err.metadata);
                                    } else {
                                        const retryTime = new Date(err.metadata.recommendedRetryTime).getTime();
                                        if (Date.now() + subscribeRetryIntervalInSeconds * 1000 < retryTime) {
                                            yield new Promise((res)=>setTimeout(res, retryTime - Date.now() - subscribeRetryIntervalInSeconds * 1000));
                                        }
                                    }
                                } else {
                                    _this._logSubscriptionError(accountId, `${instanceId}: Failed to subscribe`, err);
                                    if (err.name === 'NotFoundError') {
                                        _this.refreshAccount(accountId);
                                    }
                                    if (err.name === 'TimeoutError') {
                                        const mainAccountId = _this._websocketClient.accountsByReplicaId[accountId];
                                        if (mainAccountId) {
                                            const region = _this._websocketClient.getAccountRegion(accountId);
                                            const connectedInstances = _this._latencyService.getActiveAccountInstances(mainAccountId);
                                            // eslint-disable-next-line max-depth
                                            if (!connectedInstances.some((instance)=>instance.startsWith(`${mainAccountId}:${region}`))) {
                                                _this._timeoutErrorCounter[accountId] = _this._timeoutErrorCounter[accountId] || 0;
                                                _this._timeoutErrorCounter[accountId]++;
                                                // eslint-disable-next-line max-depth
                                                if (_this._timeoutErrorCounter[accountId] > 4) {
                                                    _this._timeoutErrorCounter[accountId] = 0;
                                                    _this.refreshAccount(accountId);
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                            resolveSubscribe();
                        });
                        return function subscribeTask() {
                            return _ref.apply(this, arguments);
                        };
                    }();
                    subscribeTask();
                    yield _this._subscriptions[instanceId].task.promise;
                    if (!_this._subscriptions[instanceId].shouldRetry) {
                        break;
                    }
                    const retryInterval = subscribeRetryIntervalInSeconds;
                    subscribeRetryIntervalInSeconds = Math.min(subscribeRetryIntervalInSeconds * 2, 300);
                    let resolve;
                    let subscribePromise = new Promise((res)=>{
                        resolve = res;
                    });
                    _this._subscriptions[instanceId].waitTask = setTimeout(()=>{
                        resolve(true);
                    }, retryInterval * 1000);
                    _this._subscriptions[instanceId].future = {
                        resolve,
                        promise: subscribePromise
                    };
                    const result = yield _this._subscriptions[instanceId].future.promise;
                    _this._subscriptions[instanceId].future = null;
                    if (!result) {
                        break;
                    }
                }
                delete _this._subscriptions[instanceId];
            }
        })();
    }
    /**
   * Unsubscribe from account
   * @param {String} accountId id of the MetaTrader account to unsubscribe
   * @param {Number} instanceNumber instance index number
   * @returns {Promise} promise which resolves when socket unsubscribed
   */ unsubscribe(accountId, instanceNumber) {
        var _this = this;
        return _async_to_generator(function*() {
            _this.cancelAccount(accountId);
            delete _this._subscriptionState[accountId];
            return _this._websocketClient.rpcRequest(accountId, {
                type: 'unsubscribe',
                instanceIndex: instanceNumber
            });
        })();
    }
    /**
   * Cancels active subscription tasks for an instance id
   * @param {String} instanceId instance id to cancel subscription task for
   */ cancelSubscribe(instanceId) {
        if (this._subscriptions[instanceId]) {
            const subscription = this._subscriptions[instanceId];
            if (subscription.future) {
                subscription.future.resolve(false);
                clearTimeout(subscription.waitTask);
            }
            if (subscription.task) {
                subscription.task.resolve(false);
            }
            subscription.shouldRetry = false;
        }
    }
    /**
   * Cancels active subscription tasks for an account
   * @param {String} accountId account id to cancel subscription tasks for
   */ cancelAccount(accountId) {
        for (let instanceId of Object.keys(this._subscriptions).filter((key)=>key.startsWith(accountId))){
            this.cancelSubscribe(instanceId);
        }
        Object.keys(this._awaitingResubscribe).forEach((instanceNumber)=>delete this._awaitingResubscribe[instanceNumber][accountId]);
        delete this._timeoutErrorCounter[accountId];
    }
    /**
   * Invoked on account timeout.
   * @param {String} accountId id of the MetaTrader account
   * @param {Number} instanceNumber instance index number
   */ onTimeout(accountId, instanceNumber) {
        const region = this._websocketClient.getAccountRegion(accountId);
        if (this._websocketClient.socketInstancesByAccounts[instanceNumber][accountId] !== undefined && this._websocketClient.connected(instanceNumber, this._websocketClient.socketInstancesByAccounts[instanceNumber][accountId], region)) {
            this._logger.debug(`${accountId}:${instanceNumber}: scheduling subscribe because of account timeout`);
            this.scheduleSubscribe(accountId, instanceNumber, true);
        }
    }
    /**
   * Invoked when connection to MetaTrader terminal terminated
   * @param {String} accountId id of the MetaTrader account
   * @param {Number} instanceNumber instance index number
   */ onDisconnected(accountId, instanceNumber) {
        var _this = this;
        return _async_to_generator(function*() {
            yield new Promise((res)=>setTimeout(res, Math.max(Math.random() * 5, 1) * 1000));
            if (_this._websocketClient.socketInstancesByAccounts[instanceNumber][accountId] !== undefined) {
                _this._logger.debug(`${accountId}:${instanceNumber}: scheduling subscribe because account disconnected`);
                _this.scheduleSubscribe(accountId, instanceNumber, true);
            }
        })();
    }
    /**
   * Invoked when connection to MetaApi websocket API restored after a disconnect.
   * @param {Number} instanceNumber instance index number
   * @param {Number} socketInstanceIndex socket instance index
   * @param {String[]} reconnectAccountIds account ids to reconnect
   */ onReconnected(instanceNumber, socketInstanceIndex, reconnectAccountIds) {
        if (!this._awaitingResubscribe[instanceNumber]) {
            this._awaitingResubscribe[instanceNumber] = {};
        }
        const socketInstancesByAccounts = this._websocketClient.socketInstancesByAccounts[instanceNumber];
        for (let instanceId of Object.keys(this._subscriptions)){
            const accountId = instanceId.split(':')[0];
            if (socketInstancesByAccounts[accountId] === socketInstanceIndex) {
                this.cancelSubscribe(instanceId);
            }
        }
        var _this = this;
        reconnectAccountIds.forEach(function() {
            var _ref = _async_to_generator(function*(accountId) {
                if (!_this._awaitingResubscribe[instanceNumber][accountId]) {
                    _this._awaitingResubscribe[instanceNumber][accountId] = true;
                    while(_this.isAccountSubscribing(accountId, instanceNumber)){
                        yield new Promise((res)=>setTimeout(res, 1000));
                    }
                    yield new Promise((res)=>setTimeout(res, Math.random() * 5000));
                    if (_this._awaitingResubscribe[instanceNumber][accountId]) {
                        delete _this._awaitingResubscribe[instanceNumber][accountId];
                        _this._logger.debug(`${accountId}:${instanceNumber}: scheduling subscribe because account reconnected`);
                        _this.scheduleSubscribe(accountId, instanceNumber);
                    }
                }
            });
            return function(accountId) {
                return _ref.apply(this, arguments);
            };
        }());
    }
    /**
   * Schedules a task to refresh the account data
   * @param {string} accountId account id
   */ refreshAccount(accountId) {
        const mainAccountId = this._websocketClient.accountsByReplicaId[accountId];
        if (mainAccountId) {
            const registry = this._metaApi._connectionRegistry;
            const rpcConnection = registry.rpcConnections[mainAccountId];
            const region = this._websocketClient.getAccountRegion(accountId);
            if (region) {
                if (rpcConnection) {
                    rpcConnection.scheduleRefresh(region);
                }
                const streamingConnection = registry.streamingConnections[mainAccountId];
                if (streamingConnection) {
                    streamingConnection.scheduleRefresh(region);
                }
            }
        }
    }
    _logSubscriptionError(accountId, message, error) {
        const primaryAccountId = this._websocketClient.accountsByReplicaId[accountId];
        const method = this._latencyService.getSynchronizedAccountInstances(primaryAccountId).length ? 'debug' : 'error';
        this._logger[method](message, error);
    }
    /**
   * Constructs the subscription manager
   * @param {MetaApiWebsocketClient} websocketClient websocket client to use for sending requests
   * @param {MetaApi} metaApi metaApi instance
   */ constructor(websocketClient, metaApi){
        this._websocketClient = websocketClient;
        this._latencyService = websocketClient.latencyService;
        this._metaApi = metaApi;
        this._subscriptions = {};
        this._awaitingResubscribe = {};
        this._subscriptionState = {};
        this._logger = LoggerManager.getLogger('SubscriptionManager');
        this._timeoutErrorCounter = {};
        this._recentlyDeletedAccounts = {};
    }
};
/**
 * Subscription manager to handle account subscription logic
 */ export { SubscriptionManager as default };

//# sourceMappingURL=data:application/json;base64,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